﻿using System;
using BMS.Facade.Data;
using DC = BMS.DataContracts;
using System.Collections.Generic;
using BMS.Utils;
using System.Linq;
using BMS.VistaIntegration.FacadeContracts.Admin;

namespace BMS.Facade.Translators
{
    public static class BMQueryTranslator
    {
        public static AdmissionInfo ToFacadeContract(this DC.AdmissionInfo serviceContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                AdmissionInfo facadeContract = new AdmissionInfo()
                {
                    Acute = serviceContract.FeeDisposition.GetDisplayName(),
                    BedAssignedDate = serviceContract.BedAssignedDate,
                    Contract = serviceContract.Contract.GetDisplayName(),
                    Entered = serviceContract.CreationDate,
                    Fee = serviceContract.Authorization.GetDisplayName(),
                    FeeComments = serviceContract.CommentFee,
                    Problem = serviceContract.Problem,
                    ReasonUsingFee = serviceContract.ReasonFee.GetDisplayName(),
                    Removed = serviceContract.RemovedDate,
                    Requested = serviceContract.RequestedDate,
                    RoomBedAssignedId = serviceContract.AssignedBedId != null ? serviceContract.AssignedBedId : null,
                    ServiceReceivingFeeCD = serviceContract.ServiceReceive,
                    TypeOfBedWard = serviceContract.TypeOfBedWard,
                    WaitingArea = serviceContract.WaitingArea.GetDisplayName(),
                    
                    //Re-Adding WaitTime change from C73035                    
                    WaitTime = serviceContract.WaitTime.Value,

                    IdAct = serviceContract.Id.extension,                    
                    ExtensionPatientId = serviceContract.PatientId == null ? null : serviceContract.PatientId.extension,
                    RootPatientId = serviceContract.PatientId == null ? null : serviceContract.PatientId.root,
                    IsInHouse = serviceContract.IsInHouse,
                    IsBedInIsolation = serviceContract.IsBedInIsolation,
                    
                    LastEditBy = serviceContract.LastEditBy,
                    Comment = serviceContract.Comment,
                    WaitTimeAlertDate = serviceContract.WaitTimeAlertDate,
                    RemainingWaitTimeAlertHours = (serviceContract.WaitTimeAlertDate == null) ? null : serviceContract.RemovedDate == null ? (int?)(serviceContract.WaitTimeAlertDate.Value - DateTime.UtcNow).TotalHours : (int?)(serviceContract.RemovedDate.Value - DateTime.UtcNow).TotalHours,
                    IsCommunityLivingCenter = serviceContract.IsCommunityLivingCenter,
                    DateOfScreening = serviceContract.DateOfScreening,
                    VeteranPreference = serviceContract.VeteranPreference,
                    MentalHealthComment = serviceContract.MentalHealthComment,
                    TransportationComment = serviceContract.TransportationComment,
                    PatientFlowComment = serviceContract.PatientFlowComment,
                    FeeAccepted = serviceContract.FeeAccepted,
                    EvacExpectedPickUp = serviceContract.EvacExpectedPickUp,
                    IsCommunityHospital = serviceContract.IsCommunityHospital,
                    TransferCoordinator = serviceContract.TransferCoordinator,
                    TransferCoordinatorPhone = serviceContract.TransferCoordinatorPhone
                };
                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static TransferInfo ToFacadeContract(this DC.TransferInfo serviceContract, List<Facility> facilities)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TimeSpan waitingTime = TimeSpan.FromSeconds(0);
                if (serviceContract.ReqCreateDateTime.HasValue)
                    waitingTime = DateTime.UtcNow.Subtract(serviceContract.ReqCreateDateTime.Value);

                TransferInfo facadeContract = new TransferInfo()
                {
                    AdmissionDate = serviceContract.LocationAdmissionDate,
                    Comment = serviceContract.Comment,
                    CurrentLocation = serviceContract.CurrentLocation,
                    Contract = serviceContract.Contract != null ? serviceContract.Contract.GetDisplayName() : String.Empty,
                    Diagnosis = serviceContract.Diagnosis,
                    Era = serviceContract.Era != null ? serviceContract.Era.displayName : String.Empty,
                    Id = serviceContract.Id,
                    PatientId = serviceContract.PatientId,
                    Specialty = serviceContract.Speciality != null ? serviceContract.Speciality.displayName : String.Empty,   
                    VisnNumber = serviceContract.FacilityVISN,
                    /*HHG: Replacing below WaitingTime logic because Dispositiondate should always be null
                        WaitingTime = serviceContract.DispositionDate == null ? DateTime.UtcNow.Subtract(serviceContract.CreationDate.Value) : serviceContract.DispositionDate.Value.Subtract(serviceContract.CreationDate.Value),
                     */
                    WaitingTime = waitingTime,
                    IsNationalWaitList = serviceContract.IsNationalWaitList,
                    TypeOfBedWard = serviceContract.TypeOfBedWard,
                    IsFee = serviceContract.IsFee,
                    TransferCoordinator = serviceContract.TransferCoordinator,
                    TransferCoordinatorPhone = serviceContract.TransferCoordinatorPhone
                };

                if (serviceContract.FacilityId != null && !serviceContract.FacilityId.extension.Equals(Guid.Empty.ToString()))
                {
                    facadeContract.FacilityFrom = facilities.Where(a => a.Id.extension.Equals(serviceContract.FacilityId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    facadeContract.FacilityCodeFrom = facadeContract.FacilityFrom != null ? ((Facility)facadeContract.FacilityFrom).Code : string.Empty;
                }
                if (serviceContract.AdmissionFacilityId != null && !serviceContract.AdmissionFacilityId.extension.Equals(Guid.Empty.ToString()))
                {
                    Facility facilityTo = facilities.Where(a => a.Id.extension.Equals(serviceContract.AdmissionFacilityId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() as Facility;
                    facadeContract.FacilityCodeTo = facilityTo != null ? facilityTo.Code : string.Empty;
                }

                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static TransferInfo ToFacadeContract(this DC.PatientWaitingStandardView serviceContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                /* HHG: Replace Wait Time calculation */
                TimeSpan waitingTime = TimeSpan.FromSeconds(0);
                if (serviceContract.DispositionDate == null)
                {
                    if (serviceContract.ReqCreateDateTime.HasValue)
                        waitingTime = DateTime.UtcNow.Subtract(serviceContract.ReqCreateDateTime.Value);
                }
                else
                {
                    if (serviceContract.EvnCreateDateTime.HasValue && serviceContract.ReqCreateDateTime.HasValue)
                        serviceContract.EvnCreateDateTime.Value.Subtract(serviceContract.ReqCreateDateTime.Value);
                }

                TransferInfo facadeContract = new TransferInfo()
                {
                    AdmissionDate = serviceContract.CreationDate,
                    Comment = serviceContract.Comment,
                    CurrentLocation = serviceContract.CurrentLocation,
                    Contract = serviceContract.Contract != null ? serviceContract.Contract.GetDisplayName() : String.Empty,
                    Diagnosis = serviceContract.Diagnosis,
                    Era = serviceContract.Era != null ? serviceContract.Era.displayName : String.Empty,
                    FacilityCodeFrom = string.Empty,
                    FacilityFrom = (serviceContract.FacilityId != null) ? new Place() { Id = serviceContract.FacilityId } : null,
                    FacilityCodeTo = string.Empty,
                    Id = serviceContract.Id,
                    Specialty = serviceContract.Speciality != null ? serviceContract.Speciality.displayName : String.Empty,   
                    PatientId = serviceContract.PatientId == null ? null : serviceContract.PatientId,
                    RegionNumber = serviceContract.RegionCode,
                    VisnNumber = serviceContract.VISNCode,
                    EventDate = serviceContract.EventDate,
                    WaitingTime = waitingTime,
                    IsNationalWaitList = serviceContract.IsNationalWaitList,
                    TypeOfBedWard = serviceContract.TypeOfBedWard,
                    IsFee = serviceContract.IsFee,
                    TransferCoordinator = serviceContract.TransferCoordinator,
                    TransferCoordinatorPhone = serviceContract.TransferCoordinatorPhone
                };
                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static DateView ToFacadeContract(this DC.PatientWaitingDateView serviceContract, List<Facility> facilities)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TimeSpan waitingTime = TimeSpan.FromSeconds(0);
                if (serviceContract.DispositionDate == null)
                {
                    if (serviceContract.ReqCreateDateTime.HasValue)
                        waitingTime = DateTime.UtcNow.Subtract(serviceContract.ReqCreateDateTime.Value);
                }
                else
                {
                    if (serviceContract.EvnCreateDateTime.HasValue && serviceContract.ReqCreateDateTime.HasValue)
                        serviceContract.EvnCreateDateTime.Value.Subtract(serviceContract.ReqCreateDateTime.Value);
                }

                DateView facadeContract = new DateView()
                {
                    RequestDate = serviceContract.EventDate,
                    EntryFacilityName = (serviceContract.FacilityId != null && !serviceContract.FacilityId.extension.Equals(Guid.Empty.ToString())) ? facilities.Where(a => a.Id.extension.Equals(serviceContract.FacilityId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().Code : string.Empty,
                    AdmittedFacilityName = (serviceContract.AdmissionFacilityId != null && !serviceContract.AdmissionFacilityId.extension.Equals(Guid.Empty.ToString())) ? facilities.Where(a => a.Id.extension.Equals(serviceContract.AdmissionFacilityId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().Code : string.Empty,
                    PatientId = serviceContract.PatientId == null ? null : serviceContract.PatientId,
                    Specialty = serviceContract.Specialty,
                    RegionNumber = serviceContract.RegionCode,
                    VisnNumber = serviceContract.VISNCode,
                    EntryDT = serviceContract.CreationDate.Value,
                    //WaitingTime = serviceContract.DispositionDate == null ? DateTime.UtcNow.Subtract(serviceContract.CreationDate.Value) : serviceContract.DispositionDate.Value.Subtract(serviceContract.CreationDate.Value),
                    /* HHG: Replace Wait Time calculation */
                    WaitingTime = waitingTime, 
                    EnterBy = serviceContract.EntryBy,
                    LastEditBy = serviceContract.LastEditBy,
                    LastEditDate = serviceContract.LastEditDate,
                    DispositionDate = serviceContract.DispositionDate,
                    FacilityFrom = (serviceContract.FacilityId != null) ? facilities.Where(a => a.Id.extension.Equals(serviceContract.FacilityId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() : null,
                    FacilityTo = (serviceContract.AdmissionFacilityId != null) ? facilities.Where(a => a.Id.extension.Equals(serviceContract.AdmissionFacilityId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() : null,
                    IsNationalWaitList = serviceContract.IsNationalWaitList
                };
                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static BedInfo ToFacadeContract(this DC.BedOccupancyInfo serviceContract, TimeZoneInfo tz, bool EMSModuleInactive, bool dischOrdModuleInactive, bool anticDischModuleInactive, bool dischAppModuleInactive, bool trsfOrdModuleInactive)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string bedAvailabilityStatusCode = string.Empty, bedCleaningStatusCode = string.Empty, dischargeStatusCode = string.Empty, levelOfCareCode = string.Empty;
                int levelOfCareDays = 0;
                int levelOfCareChangeDays = 0;

                //get the bed availability status concept with the icon property
                if (serviceContract.BedAvailabilityStatus != null)
                {
                    string code = string.Empty;
                    if (serviceContract.BedAvailabilityStatus == Constants.BED_OUT_OF_SERVICE_TYPE_CODE)
                    {
                        if (serviceContract.IsUnavailableFromVista.HasValue && serviceContract.IsUnavailableFromVista.Value)
                            code = Constants.BED_OOS_FROM_VISTA;
                        else
                            code = Constants.BED_OOS_FROM_BEDBOARD;
                    }
                    else if (serviceContract.BedAvailabilityStatus == Constants.BED_IN_ISOLATION_TYPE_CODE)
                        code = Constants.BED_AVAILABILITY_STATUS_ISOLATION_CODE;
                    else
                        code = string.Empty;
                    if (!string.IsNullOrEmpty(code))
                        bedAvailabilityStatusCode = code;
                    else
                        bedAvailabilityStatusCode = serviceContract.BedAvailabilityStatus;
                }

                //get the bed cleaning status concept with the icon property
                if (!EMSModuleInactive)
                {
                    if (serviceContract.BedCleanRequestedDate.HasValue)
                    {
                        if (serviceContract.BedCleanAcceptedDate.HasValue && serviceContract.BedCleanCompletedDate.HasValue)
                            bedCleaningStatusCode = Constants.BED_CLEANED;
                        else if (serviceContract.BedCleanAcceptedDate.HasValue && !serviceContract.BedCleanCompletedDate.HasValue)
                            bedCleaningStatusCode = Constants.CLEANING_BED;
                        else
                            bedCleaningStatusCode = Constants.BED_NEEDS_CLEAN;
                    }
                }

                //get the interward transfer concept with the icon property
                if (serviceContract.InterWardTransfer != null && !trsfOrdModuleInactive)
                    dischargeStatusCode = Constants.INTERNAL_TRANSFER;
                
                //get the discharge status concept with the icon property                
                if (serviceContract.DischargeStatus != null)
                {
                    if (serviceContract.DischargeStatus.ToUpper().Equals("DISCHARGE ORDER") && !dischOrdModuleInactive)
                        dischargeStatusCode = Constants.DISCHARGE_ORDER;
                    if ((serviceContract.DischargeStatus.ToUpper().Equals("ANTICIPATED DISCHARGE ORDER") && !anticDischModuleInactive) ||
                        (serviceContract.DischargeStatus.ToUpper().Equals("DISCHARGE APPOINTMENT") && !dischAppModuleInactive) )
                        dischargeStatusCode = Constants.DISCHARGE_ANTICIPATED;

                }

                //get the level of care concept with the icon property
                if (serviceContract.PatientId != null && !string.IsNullOrEmpty(serviceContract.PatientId.extension) && !serviceContract.PatientId.extension.Equals(Guid.Empty.ToString()))
                {
                    if (string.IsNullOrEmpty(serviceContract.LevelOfCare))
                        levelOfCareCode = Constants.NUMI_LEVEL_OF_CARE_STAT;
                    else if (serviceContract.LevelOfCare == "1")
                        levelOfCareCode = Constants.NUMI_LEVEL_OF_CARE_YES;
                    else if (serviceContract.LevelOfCare == "0")
                        levelOfCareCode = Constants.NUMI_LEVEL_OF_CARE_NO;

                    if (serviceContract.LevelOfCareDate.HasValue && serviceContract.AdmissionDate.HasValue && serviceContract.MovementDate.HasValue && (!levelOfCareCode.Equals(Constants.NUMI_LEVEL_OF_CARE_STAT, StringComparison.InvariantCultureIgnoreCase)))
                    {
                        DateTime levelOfCareDate = TimeZoneInfo.ConvertTimeFromUtc(serviceContract.LevelOfCareDate.Value, tz).Date;
                        DateTime movementDate = TimeZoneInfo.ConvertTimeFromUtc(serviceContract.MovementDate.Value, tz).Date;

                        if (((serviceContract.MovementDate != serviceContract.AdmissionDate) && (levelOfCareDate <= movementDate)) ||
                        ((serviceContract.MovementDate == serviceContract.AdmissionDate) && (levelOfCareDate < movementDate))) levelOfCareCode = Constants.NUMI_LEVEL_OF_CARE_STAT;
                        else levelOfCareDays = (TimeZoneInfo.ConvertTimeFromUtc(DateTime.UtcNow, tz).Date - levelOfCareDate).Days;
                    }

                    if (serviceContract.LevelOfCareChangeDate.HasValue)
                    {
                        DateTime levelOfCareChangeDate = TimeZoneInfo.ConvertTimeFromUtc(serviceContract.LevelOfCareChangeDate.Value, tz).Date;
                        levelOfCareChangeDays = (TimeZoneInfo.ConvertTimeFromUtc(DateTime.UtcNow, tz).Date - levelOfCareChangeDate).Days;
                    }

                    // comment next line if G/R is valid without a valid LOC date
                    else levelOfCareCode = Constants.NUMI_LEVEL_OF_CARE_STAT;
                }

                BedInfo facadeContract = new BedInfo()
                {
                    BedAvailabilityStatusCode = bedAvailabilityStatusCode,
                    BedCleaningStatusCode = bedCleaningStatusCode,
                    BedId = serviceContract.BedId,
                    AdmissionId = serviceContract.Id,
                    DischargeStatusCode = dischargeStatusCode,
                    LosAdmin = serviceContract.LosAdmin,
                    LosDrg = serviceContract.LosDrg,
                    LosWard = serviceContract.LosWard,
                    PatientId = serviceContract.PatientId,
                    Reason = serviceContract.Reason,
                    WardId = serviceContract.WardId,
                    LevelOfCareCode = levelOfCareCode,
                    IconFlagIdsOfBedList = serviceContract.IconFlagIdsOfBedList,
                    IconFlagIdsOfPatientList = serviceContract.IconFlagIdsOfPatientList,
                    IconFlagFacilityIdsOfBedList = serviceContract.IconFlagFacilityIdsOfBedList,
                    IconFlagFacilityIdsOfPatientList = serviceContract.IconFlagFacilityIdsOfPatientList,
                    WaitingListPatientId = serviceContract.WaitingListPatientId,
                    WaitingListInfo = (serviceContract.WaitingListInfo != null) ? serviceContract.WaitingListInfo : string.Empty,
                    FlowId = serviceContract.FlowId,
                    LevelOfCareDays = levelOfCareDays,
                    LevelOfCareChangeDays = levelOfCareChangeDays,
                    IsEvacuationPatient = serviceContract.IsEvacuationPatient,
                    EvacuationDispositionStatus = serviceContract.EvacuationDispositionStatus
                };
                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static string GetDisplayName(this InfoWorld.HL7.ITS.CD concept)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return concept != null ? concept.displayName : string.Empty;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static BedCleanInfo ToFacadeContract(this DC.BedCleanInfo serviceContract, IconInfo bedInIsolationIcon, IconInfo emsNotifyIcon)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DateTime date72 = DateTime.UtcNow.AddHours(-72);
                DateTime requestedDate72 = serviceContract.RequestedDate.Value.AddHours(72);
                BedCleanInfo facadeContract = new BedCleanInfo();

                facadeContract.BedCleanId = serviceContract.Id;
                //Manual Request
                if (serviceContract.ManualRequest != null && serviceContract.ManualRequest.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase))
                {
                    if (serviceContract.ManualRequest.displayName.ToLower().Contains(Constants.Stat.ToLower()))
                        facadeContract.ManualRequest = "Manual/" + Constants.Stat;
                    else
                        facadeContract.ManualRequest = "Manual";
                }
                else
                    facadeContract.ManualRequest = serviceContract.EventIen;

                facadeContract.Bed = (serviceContract.BedId != null) ? new Bed() { Id = serviceContract.BedId } : null;
                facadeContract.Ward = (serviceContract.WardId != null) ? new Ward() { Id = serviceContract.WardId } : null;
                facadeContract.WardName = string.Empty;
                facadeContract.RequestedDate = serviceContract.RequestedDate;
                facadeContract.VacatedDate = serviceContract.VacatedDate.HasValue ? serviceContract.VacatedDate : serviceContract.RequestedDate;
                if (!serviceContract.CompletedDate.HasValue && !serviceContract.AcceptedDate.HasValue && serviceContract.RequestedDate < date72)
                {
                    facadeContract.AcceptedBy = "BMS BED CLEAN has been accepted by system.";
                    facadeContract.AcceptedDate = requestedDate72;
                }
                else
                {
                    facadeContract.AcceptedBy = serviceContract.AcceptedBy;
                    facadeContract.AcceptedDate = serviceContract.AcceptedDate;
                }
                if (!serviceContract.CompletedDate.HasValue && serviceContract.RequestedDate < date72)
                {
                    facadeContract.CompletedBy = "BMS BED CLEAN has been completed by system.";
                    facadeContract.CompletedDate = requestedDate72;
                    facadeContract.LastEditBy = "BMS system.";
                    facadeContract.LastEditDate = requestedDate72;
                    facadeContract.Comment = "More than 72 hours have passed since request date.";
                    facadeContract.CommentedBy = "BMS BED CLEAN has been commented by system.";
                    facadeContract.CommentedDate = requestedDate72;
                }
                else
                {
                    facadeContract.CompletedBy = serviceContract.CompletedBy;
                    facadeContract.CompletedDate = serviceContract.CompletedDate;
                    facadeContract.LastEditBy = serviceContract.LastEditedBy;
                    facadeContract.LastEditDate = serviceContract.LastEditDate;
                    facadeContract.Comment = serviceContract.Comment;
                    facadeContract.CommentedBy = serviceContract.CommentedBy;
                    facadeContract.CommentedDate = serviceContract.CommentedDate;
                }

                //Unavailable Type
                if (serviceContract.UnavailableType != null && serviceContract.UnavailableType.code.Equals(Constants.BED_IN_ISOLATION_TYPE_CODE))
                    facadeContract.UnavailableType = bedInIsolationIcon;
                else
                    facadeContract.UnavailableType = null;
                // EMS Notified
                if (serviceContract.EMSNotify != null && serviceContract.EMSNotify.ToLower().Equals(Constants.Yes.ToLower()))
                    facadeContract.EMSNotify = emsNotifyIcon;
                else
                    facadeContract.EMSNotify = null;

                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static JobLogInfo ToFacadeContract(this DC.JobLogInfo serviceContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new JobLogInfo()
                {
                    EndDate = serviceContract.EndDate,
                    ErrorMessage = serviceContract.ErrorMessage,
                    Details = serviceContract.Details,
                    Index = serviceContract.Index,
                    LaunchType = serviceContract.LaunchType,
                    Parameters = new Dictionary<string, object>() { { "StartDate", serviceContract.QueryStartDate }, { "EndDate", serviceContract.QueryEndDate } },
                    RetrievalMethod = serviceContract.RetrievalMethod,
                    RetrievedDataType = (BMS.VistaIntegration.FacadeContracts.Admin.VistaDataType)serviceContract.RetrievedDataType,
                    RetrievedEntriesNumber = serviceContract.RetrievedEntriesNumber,
                    StartDate = serviceContract.StartDate,
                    Status = serviceContract.Status,
                    TotalCount = serviceContract.TotalCount,
                    VistaId = serviceContract.VistaId,
                    VistaName = serviceContract.VistaName
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static IconInfo ToFacadeContract(this DC.IconInfo serviceContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IconInfo facadeContract = new IconInfo()
                {
                    Active = serviceContract.Active,
                    Facility = serviceContract.FacilityId != null ? new Facility() { Id = serviceContract.FacilityId } : null,
                    Group = serviceContract.Group,
                    IconDescription = serviceContract.IconDescription,
                    IconId = serviceContract.IconId,
                    IconName = serviceContract.IconName,
                    ImageBytes = serviceContract.ImageBytes,
                    ImageBytesInactive = serviceContract.ImageBytesInactive,
                    MouseOverText = serviceContract.MouseOverText,
                    AutoIconPattern = serviceContract.AutoIconPattern,
                    AutoIconActive = serviceContract.AutoIconActive,
                    IsIsolation = serviceContract.IsIsolation,
                    OrderIndex = serviceContract.OrderIndex,
                    PatientOrRoomBed = serviceContract.PatientOrRoomBed,
                    Subgroup = serviceContract.Subgroup,
                    IsAssociation = serviceContract.IsAssociation,
                    IconType = serviceContract.IconType,
                    Published = serviceContract.Published,
                    ImageName = serviceContract.ImageName
                };
                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static IconAssociation ToFacadeContract(this DC.IconAssociation serviceContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IconAssociation facadeContract = new IconAssociation()
                {
                    Bed = serviceContract.BedId != null ? new Place() { Id = serviceContract.BedId } : null,
                    Patient = null,
                    Facility = null,
                    ListIconFlagFacilityIdsOfBed = serviceContract.ListIconFlagFacilityIdsOfBed,
                    ListIconFlagFacilityIdsOfPatient = serviceContract.ListIconFlagFacilityIdsOfPatient,
                    ListIconFlagIdsOfBed = serviceContract.ListIconFlagIdsOfBed,
                    ListIconFlagIdsOfPatient = serviceContract.ListIconFlagIdsOfPatient,
                    WaitingListPatientId = serviceContract.WaitingListPatientId,
                    WaitingListInfo = (serviceContract.WaitingListInfo != null) ? serviceContract.WaitingListInfo : string.Empty
                };
                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static PatientIcon ToFacadeContract(this DC.PatientIcon serviceContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                PatientIcon facadeContract = new PatientIcon()
                {
                    PatientId = Guid.Parse(serviceContract.PatientId.extension),
                    IconFlagIds = serviceContract.IconFlagIds,
                };
                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static WaitingListReport ToFacadeContract(this DC.WaitingListReport serviceContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WaitingListReport facadeContract = new WaitingListReport()
                {
                    ActId = serviceContract.ActId,
                    AuthorizedFee = serviceContract.AuthorizedFee,
                    BedAssignedDate = serviceContract.BedAssignedDate,
                    BedRequestedDate = serviceContract.BedRequestedDate,
                    CommunityServices = serviceContract.CommunityServices,
                    ContractFee = serviceContract.ContractFee,
                    DispositionFee = serviceContract.DispositionFee,
                    EnteredDate = serviceContract.EnteredDate,
                    FeeComments = serviceContract.FeeComments,
                    InHouseTransfer = serviceContract.InHouseTransfer,
                    LastEditedBy = serviceContract.LastEditedBy,
                    PatientId = serviceContract.PatientId,
                    PresentingProblem = serviceContract.PresentingProblem,
                    ReasonUsingFee = serviceContract.ReasonUsingFee,
                    RemovedDate = serviceContract.RemovedDate,
                    RequestedDate = serviceContract.RequestedDate,
                    RoomBedAssignedId = serviceContract.RoomBedAssignedId,
                    TypeOfBedWard = serviceContract.TypeOfBedWard,
                    WaitingArea = serviceContract.WaitingArea
                };
                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static EvacuationPatient ToFacadeContract(this DC.EvacuationPatient serviceContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (serviceContract == null)
                    return null;
                EvacuationPatient facadeContract = new EvacuationPatient()
                {
                    CanceledDate = serviceContract.CanceledDate,
                    CreationDate = serviceContract.CreationDate,
                    Id = serviceContract.Id,
                    Parent = (serviceContract.Parent == null) ? null : serviceContract.Parent.ToFacadeContract(),
                    PatientId = serviceContract.PatientId,
                    IEN = serviceContract.IEN,
                    FlowId = serviceContract.FlowId
                };
                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static BedStaffHistory ToFacadeContract(this DC.BedStaffHistory serviceContract)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (serviceContract == null)
                    return null;
                BedStaffHistory facadeContract = new BedStaffHistory()
                {
                    Bed = serviceContract.BedId != null ? new Place() { Id = serviceContract.BedId } : null,
                    CreatedBy = serviceContract.CreatedBy,
                    CreationDate = serviceContract.CreationDate,
                    Staff = serviceContract.Staff,
                    TransactionTypeCode = serviceContract.TransactionTypeCode
                };
                return facadeContract;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
